# -*- coding: utf-8 -*-
'''
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
'''

import sys
import os
import platform
import shutil
import struct
import json
import logging

import pandas as pd

from PySide6 import QtCore, QtWidgets
from PySide6.QtGui import QIcon, QFont, QPixmap, QAction
from PySide6.QtWidgets import QTableWidgetItem, QComboBox, QMessageBox, QItemDelegate, QFileDialog

from ui import Ui_ddrbin
from ui import Ui_textWindow
#from lib import modify_ddr_bin
from lib.modify_ddr_bin import *

window_title_g = "Rockchip DDR bin tool V1.05.1"
logo_file_g = "resource/image/logo.icon"
logo_file_ubuntu_g = "resource/image/logo.icon"
logo_file_windows_g = "resource/image/logo.icon"
rockchip_logo_file_g = "resource/image/rockchip_logo.png"

# logging配置
logger = logging.getLogger()
logger.setLevel(level=logging.DEBUG)

class EmptyDelegate(QItemDelegate):
  def __init__(self,parent):
    super(EmptyDelegate, self).__init__(parent)
  def createEditor(self, QWidget, QStyleOptionViewItem, QModelIndex):
    return None

# 禁用QComboBox的鼠标滚动选值功能
class QComboBox(QComboBox):
    def wheelEvent(self, QWheelEvent):
        pass

class login_textWindow(QtWidgets.QWidget, Ui_textWindow):
    def __init__(self):
        super(login_textWindow,self).__init__()
        self.setupUi(self)

#ddrparam界面
class login_ddrbin(QtWidgets.QMainWindow, Ui_ddrbin):
    def __init__(self):
        super(login_ddrbin,self).__init__()
        self.setupUi(self)
        self.setup_ui_setting()

        # logging配置
        self.logfile = "log.txt"
        handler = logging.FileHandler(self.logfile,mode='w',encoding='utf-8')
        formatter = logging.Formatter("%(asctime)s - %(filename)s(%(funcName)s)[line:%(lineno)d] - %(levelname)s: %(message)s")
        handler.setFormatter(formatter)
        console = logging.StreamHandler() # 输出到console的handler
        console.setLevel(logging.DEBUG)
        logger.addHandler(handler)
        logger.addHandler(console)

        logger.debug(window_title_g)
        #logger.info('This is an info message.')
        #logger.warning('This is a warning message.')
        #logger.error('This is an error message.')
        #logger.critical('This is a critical message.')

    def open_text(self, q):
        if q.text() == "ddrbin_tool 使用指南":
            file_path = "./resource/doc/ddrbin_tool_user_guide.txt"
        elif q.text() == "版本说明":
            file_path = "./resource/doc/ddbin_release_note.txt"
        else:
            return
        self.textWindow = login_textWindow()
        self.textWindow.show()

        with open(file_path, "r", encoding='utf-8') as f:
            text = f.read()
            self.textWindow.textBrowser.setText(text)
        self.textWindow.textBrowser.setFont(QFont('Times', 10))
    # 支持UI配置重置
    def setup_ui_setting(self):
        # 显示窗口标题
        self.setWindowTitle(window_title_g)
        # 显示窗口图标
        if platform.system().lower() == 'linux':
            self.setWindowIcon(QIcon(logo_file_ubuntu_g))
        else:
            self.setWindowIcon(QIcon(logo_file_windows_g))
        # 注册状态栏
        self.statusBar()
        self.statusBar().showMessage('请选择CHIP型号')

        # rockchip logo
        pixmap = QPixmap(rockchip_logo_file_g)
        self.label_rk_logo.setPixmap (pixmap)
        self.label_rk_logo.setScaledContents (True)

        # 注册reset按钮功能，重置UI配置
        self.pushButton_reset.clicked.connect(self.clicked_button_reset)

        # 登录界面，芯片平台选择
        # 从json文件中读取参数
        self.ddrbin_param_dict = self.get_params_from_json("./resource/config/ddrbin_tool.json")[0]
        self.loader_pack_dict = self.get_params_from_json("./resource/config/ddrbin_tool.json")[1]
        chip_list = self.ddrbin_param_dict.keys()
        self.comboBox_chipName.addItems(chip_list)
        self.comboBox_chipName.setFont(QFont('Times', 12, QFont.Black))
        # 获取chip name
        self.chip_name = self.comboBox_chipName.currentText()
        self.comboBox_chipName.currentIndexChanged.connect(self.chip_name_change)

        # 登录界面，语言选择
        language_list=["中文", "English"]
        self.comboBox_language.addItems(language_list)
        self.comboBox_language.currentIndexChanged.connect(self.language_change)

        # 变量声明
        self.chip_name = 'null'
        self.ddr_bin_file_path = 'null'
        self.loader_file_path = 'null'
        self.loader_pack_params = 'null'
        self.language = 'null'
        self.res_dict = {}
        self.res_dict_diff = {}
        self.loader_params_dict = {}

        # 按钮注册和属性配置
        # 注册ddr bin相关功能按钮
        self.pushButton_db_sel.clicked.connect(self.clicked_button_db_sel)
        self.pushButton_db_con.clicked.connect(self.clicked_button_db_con)
        self.pushButton_db_gen.clicked.connect(self.clicked_button_db_gen)
        # 注册loader相关功能按钮
        #self.pushButton_ld_sel.clicked.connect(self.clicked_button_ld_sel)
        #self.pushButton_ld_con.clicked.connect(self.clicked_button_ld_con)
        self.pushButton_ld_gen.clicked.connect(self.clicked_button_ld_gen)
        # 注册DDR类型选择按钮
        self.pushButton_common.clicked.connect(self.clicked_button_common)
        self.pushButton_lpddr5.clicked.connect(self.clicked_button_lpddr5)
        self.pushButton_lpddr4.clicked.connect(self.clicked_button_lpddr4)
        self.pushButton_lpddr4x.clicked.connect(self.clicked_button_lpddr4x)
        self.pushButton_lpddr3.clicked.connect(self.clicked_button_lpddr3)
        self.pushButton_lpddr2.clicked.connect(self.clicked_button_lpddr2)
        self.pushButton_ddr4.clicked.connect(self.clicked_button_ddr4)
        self.pushButton_ddr3.clicked.connect(self.clicked_button_ddr3)
        self.pushButton_ddr2.clicked.connect(self.clicked_button_ddr2)
        self.pushButton_all.clicked.connect(self.clicked_button_all)
        # 隐藏loader相关按钮
        #self.pushButton_ld_sel.setVisible(False)
        #self.pushButton_ld_con.setVisible(False)
        self.pushButton_ld_gen.setVisible(False)
        #self.lineEdit_loader.setVisible(False)
        self.lineEdit_pack_param.setVisible(False)

        # 隐藏 all param按钮
        self.pushButton_all.setVisible(False)

        # disable ddr bin generate按钮
        self.pushButton_db_gen.setEnabled(False)
        # disable DDR类型选择按钮
        self.pushButton_all.setEnabled(False)
        self.pushButton_common.setEnabled(False)
        self.pushButton_lpddr5.setEnabled(False)
        self.pushButton_lpddr4.setEnabled(False)
        self.pushButton_lpddr4x.setEnabled(False)
        self.pushButton_lpddr3.setEnabled(False)
        self.pushButton_lpddr2.setEnabled(False)
        self.pushButton_ddr4.setEnabled(False)
        self.pushButton_ddr3.setEnabled(False)
        self.pushButton_ddr2.setEnabled(False)
        # 在用户选中CHIP类型前，disable功能按钮
        self.pushButton_db_con.setEnabled(False)
        self.pushButton_db_sel.setEnabled(False)
        self.lineEdit_ddrbin.setEnabled(False)
        self.radioButton_overWrite.setEnabled(False)
        self.radioButton_saveCfg.setEnabled(False)
        self.lineEdit_chipOther.setVisible(False)

        # 调用Drops方法
        self.setAcceptDrops(True)

        # 添加菜单栏和工具栏
        help_menu = self.menuBar().addMenu("Help")
        help_menu.addAction("ddrbin_tool 使用指南")
        help_menu.addAction("版本说明")
        help_menu.triggered[QAction].connect(self.open_text)

        #self.file_menu_text_path = "./resource/doc/ddrbin_tool_user_guide.txt"
        #text_action.triggered.connect(self.open_text)
    # 鼠标拖入事件
    def dragEnterEvent(self, evn):
        #print('鼠标拖入窗口了')
        # 鼠标放开函数事件
        evn.accept()
    # 鼠标放开执行
    def dropEvent(self, evn):
        #print('鼠标放开了')
        logger.debug('拖入文件路径：' + evn.mimeData().text())
        if "Unknown" in self.comboBox_chipName.currentText():
            if self.language == "English":
                self.show_warning_message('Please select CHIP type first')
            else:
                self.show_warning_message('请先选择CHIP型号')
            return

        # 获取鼠标拖入文件的路径
        if platform.system().lower() == 'linux':
            file_fp = str(evn.mimeData().text()[7:-2].replace('%20', ' '))
        else:
            # 本地文件路径 file:///C:/Users/
            # 服务器文件路径 file://10.10.10.164/
            if str(evn.mimeData().text()[:8]) == "file:///":
                file_fp = str(evn.mimeData().text()[8:])
            else:
                file_fp = str(evn.mimeData().text()[5:])

        # 拖入文件时，文件名特殊字符处理
        if os.path.exists(file_fp) == False:
            file_fp=file_fp.replace('%20', ' ')
            file_fp=file_fp.replace('%23', '#')
            file_fp=file_fp.replace('%25', '%')
            file_fp=file_fp.replace('%5E', '^')

        logger.debug("file_fp %s"%(file_fp))
        self.lineEdit_ddrbin.setText(file_fp)

        if self.pushButton_db_con.isEnabled():
            self.clicked_button_db_con()
        else:
            self.clicked_button_ld_con()
    #def dragMoveEvent(self, evn):
    #    print('鼠标移入')
    # 弹出提示框
    def show_warning_message(self, message):
        QMessageBox.warning(self, "Warning", message, QMessageBox.Ok)
    def show_information_message(self, message):
        QMessageBox.information(self, "Information", message, QMessageBox.Ok)
    def show_error_message(self, message):
        QMessageBox.critical(self, "Error", message)
    # 从json文件中读取配置参数
    def get_params_from_json(self, file_path):
        # 打开并读取json文件内容
        with open(file_path, encoding="utf-8") as fp:
            params_list = json.load(fp)
        return params_list
    def language_change(self):
        # 主界面语言设置
        self.language = self.comboBox_language.currentText()
        if self.language == "English":
            self.lineEdit_ddrbin_text = self.lineEdit_ddrbin.placeholderText()
            self.lineEdit_ddrbin.setPlaceholderText("Select ddr bin file")
            self.lineEdit_chipOther.setPlaceholderText("Input chip info")
            self.pushButton_db_con_text = self.pushButton_db_con.text()
            self.pushButton_db_con.setText("submit")
            self.pushButton_db_gen_text = self.pushButton_db_gen.text()
            self.pushButton_db_gen.setText("Create ddr bin")
            self.pushButton_ld_gen_text = self.pushButton_ld_gen.text()
            self.pushButton_ld_gen.setText("Create loader")
            self.radioButton_overWrite_text = self.radioButton_overWrite.text()
            self.radioButton_overWrite.setText("Overwrite File")
            self.radioButton_saveCfg_text = self.radioButton_saveCfg.text()
            self.radioButton_saveCfg.setText("save config")
            self.statusBar().showMessage('Please select CHIP type')
        else:
            self.lineEdit_ddrbin.setPlaceholderText(self.lineEdit_ddrbin_text)
            self.pushButton_db_con.setText(self.pushButton_db_con_text)
            self.pushButton_db_gen.setText(self.pushButton_db_gen_text)
            self.pushButton_ld_gen.setText(self.pushButton_ld_gen_text)
            self.radioButton_overWrite.setText(self.radioButton_overWrite_text)
            self.radioButton_saveCfg.setText(self.radioButton_saveCfg_text)
            self.statusBar().showMessage('请选择CHIP型号')
            self.lineEdit_chipOther.setPlaceholderText("请输入芯片名")
    # 获取chip_name
    def chip_name_change(self):
        self.chip_name = self.comboBox_chipName.currentText()
        if "Unknown" not in self.chip_name:
            # 选择chip_name后，Enable功能按钮
            self.pushButton_db_con.setEnabled(True)
            self.pushButton_db_sel.setEnabled(True)
            self.lineEdit_ddrbin.setEnabled(True)
            self.radioButton_overWrite.setEnabled(True)
            self.radioButton_saveCfg.setEnabled(True)
            if "Other" == self.chip_name:
                self.lineEdit_chipOther.setVisible(True)
            else:
                self.lineEdit_chipOther.setVisible(False)
        else:
            # chip_name为Unknown时，Disable功能按钮
            self.pushButton_db_con.setEnabled(False)
            self.pushButton_db_sel.setEnabled(False)
            self.lineEdit_ddrbin.setEnabled(False)
            self.radioButton_overWrite.setEnabled(False)
            self.radioButton_saveCfg.setEnabled(False)
            self.lineEdit_chipOther.setVisible(False)

        if self.language == "English":
            self.statusBar().showMessage('Ready: clicked [...] to select ddr bin file')
        else:
            self.statusBar().showMessage('Ready: 请点击【...】选择要修改的文件或者输入文件路径后点击【确认】')
    # 重置UI配置
    def clicked_button_reset(self):
        self.setupUi(self)
        self.setup_ui_setting()
    # 刷新DDR参数tableWidget
    def show_ddr_params_tableWidget(self, csv_file_path, dram_type = "NULL"):
        # 判断存放DDR params的字典是否存在
        if not bool(self.res_dict):
            return

        # 读取完整的ddr params数据
        data_df_full = modify_ddr_bin.get_cfg_file(self.ddrbin_param_dict["Other"]).fillna(' ')

        # 截取指定DDR类型的数据，如果是"DRAM_TYPE_ALL"，则显示完整的ddr params数据
        if dram_type == "DRAM_TYPE_ALL":
            data_df = data_df_full
            columns_list = data_df.columns.tolist()
            data_df = data_df.loc[(data_df[columns_list[0]]!=' ')]
        else:
            # 读取当前芯片平台的ddr params数据
            data_df = modify_ddr_bin.get_cfg_file(csv_file_path).fillna(' ')
            columns_list = data_df.columns.tolist()
            # 为空的项，从data_df_full读取填充
            for row in range(data_df.shape[0]):
                if data_df.iloc[row, 2] == ' ' and data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 2] != ' ':
                    data_df.iloc[row, 2] = data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 2]
                if data_df.iloc[row, 3] == ' ' and data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 3] != ' ':
                    data_df.iloc[row, 3] = data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 3]
                if data_df.iloc[row, 4] == ' ' and data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 4] != ' ':
                    data_df.iloc[row, 4] = data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 4]
                if data_df.iloc[row, 5] == ' ' and data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 5] != ' ':
                    data_df.iloc[row, 5] = data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 5]
                if data_df.iloc[row, 6] == ' ' and data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 6] != ' ':
                    data_df.iloc[row, 6] = data_df_full.loc[data_df_full[columns_list[0]] == str(data_df.iloc[row, 0])].iloc[0, 6]
            # 截取自定DDR类型的数据
            data_df = data_df.loc[(data_df[columns_list[-1]] == dram_type)]

        # 删除最后一列(属性)
        data_df = data_df.drop([columns_list[-1]], axis =1)
        if self.language == "English":
            data_df = data_df.drop([columns_list[-3]], axis =1)
        else:
            data_df = data_df.drop([columns_list[-2]], axis =1)
        # 获取新的datafram列名称
        if self.language == "English":
            columns_list = ["name", "value", "effective value", "Unit", "params"]
        else:
            columns_list = data_df.columns.tolist()
        # 设置列表模型
        self.tableWidget.setColumnCount(len(columns_list))
        self.tableWidget.setRowCount(data_df.shape[0])
        # 设置水平方向两个头标签文本内容
        self.tableWidget.setHorizontalHeaderLabels(columns_list)

        # 从data_df读取数据填充tableWidget item
        for row in range(data_df.shape[0]):
            for column in range(data_df.shape[1]):
                #print(row, column)
                # 如果“有效值”带“|”，则“配置值”显示为QComboBox复选框形式
                if column == 1:
                    try:
                        item_data = self.res_dict[str(data_df.iloc[row, 0])]
                    except KeyError:
                        continue
                    newItem = QTableWidgetItem(item_data)
                    # 设置字体，字号，颜色
                    newItem.setFont(QFont('Times', 12, QFont.Black))
                    self.tableWidget.setItem(row, column, newItem)
                    # 判断有效值带“|”，则创建QComboBox()
                    data_str = str(data_df.iloc[row, column + 1])
                    if "|" in data_str:
                        data_list = data_str.split('|')
                        if self.tableWidget.item(row, column).text() not in data_list:
                            data_list.append(self.tableWidget.item(row, column).text())
                        comBox_direction = QComboBox()
                        comBox_direction.addItems(data_list)
                        comBox_direction.setCurrentText(self.tableWidget.item(row, column).text())
                        #comBox_direction.setStyleSheet("QComboBox{margin:3px};")
                        # 设置字体，字号，颜色
                        comBox_direction.setFont(QFont('Times', 12, QFont.Black))
                        self.tableWidget.setCellWidget(row, column, comBox_direction)
                    # 判断有效值带"——"，则设置单元格不可编辑
                    elif "——" in data_str:
                        newItem.setFlags(QtCore.Qt.ItemIsEnabled)
                # 如果“有效值”带“|”，因为“配置值”项已改为复选框形式，“有效值”项目值显示单字符“|”
                elif column == 2:
                    item_data = str(data_df.iloc[row, column])
                    if "|" in item_data:
                        newItem = QTableWidgetItem('|')
                    else:
                        newItem = QTableWidgetItem(item_data)
                    self.tableWidget.setItem(row, column, newItem)
                else:
                    item_data = str(data_df.iloc[row, column])
                    newItem = QTableWidgetItem(item_data)
                    self.tableWidget.setItem(row, column, newItem)

        # 水平方向标签拓展剩下的窗口部分，填满表格
        self.tableWidget.horizontalHeader().setStretchLastSection(True)
        # 水平方向，根据内容自动调节大小
        self.tableWidget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        # 垂直方向，根据内容自动调节大小
        self.tableWidget.verticalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        # 隔行改变颜色
        self.tableWidget.setAlternatingRowColors(True)
        # 除第一列外，其他列设置为不可编辑
        for column in range(self.tableWidget.columnCount()):
            if column != 1:
                self.tableWidget.setItemDelegateForColumn(column, EmptyDelegate(self))


    # 根据ddr params，enable相关按钮
    def enable_pushbutton_dram_type(self, data_dict):
        if not bool(data_dict):
            return

        self.pushButton_db_gen.setEnabled(True)
        self.pushButton_ld_gen.setEnabled(True)

        self.pushButton_common.setEnabled(True)
        self.pushButton_all.setEnabled(True)
        self.pushButton_lpddr5.setEnabled(False)
        self.pushButton_lpddr4.setEnabled(False)
        self.pushButton_lpddr4x.setEnabled(False)
        self.pushButton_lpddr3.setEnabled(False)
        self.pushButton_lpddr2.setEnabled(False)
        self.pushButton_ddr4.setEnabled(False)
        self.pushButton_ddr3.setEnabled(False)
        self.pushButton_ddr2.setEnabled(False)
        if data_dict["lp5_freq"] != '0':
            self.pushButton_lpddr5.setEnabled(True)
        if data_dict["lp4_freq"] != '0':
            self.pushButton_lpddr4.setEnabled(True)
        if data_dict["lp4x_freq"] != '0':
            self.pushButton_lpddr4x.setEnabled(True)
        if data_dict["lp3_freq"] != '0':
            self.pushButton_lpddr3.setEnabled(True)
        if data_dict["lp2_freq"] != '0':
            self.pushButton_lpddr2.setEnabled(True)
        if data_dict["ddr4_freq"] != '0':
            self.pushButton_ddr4.setEnabled(True)
        if data_dict["ddr3_freq"] != '0':
            self.pushButton_ddr3.setEnabled(True)
        if data_dict["ddr2_freq"] != '0':
            self.pushButton_ddr2.setEnabled(True)
    # 从tableWidget读取配置值更新ddr params
    def update_ddr_params_by_tableWidget(self, remove_CellWidget = True):
            try:
                self.tableWidget
            except AttributeError:
                return

            # 将复选框选中的值，填入tableWidget item
            for row in range(self.tableWidget.rowCount()):
                #for column in range(self.tableWidget.columnCount()):
                    #print(self.tableWidget.item(row, 0).text())
                    try:
                        item_str = self.tableWidget.item(row, 2).text()
                    except AttributeError:
                        continue
                    # 读取复选框中的值
                    if "|" in item_str:
                        newItem = QTableWidgetItem(self.tableWidget.cellWidget(row, 1).currentText())
                        self.tableWidget.setItem(row, 1, newItem)
                        if remove_CellWidget == True:
                            self.tableWidget.removeCellWidget(row, 1)
                    # 检查配置值是否在有效范围内
                    elif "-" in item_str:
                        offset = item_str.find('-')
                        try:
                            value = self.tableWidget.item(row, 1).text()
                        except AttributeError:
                            continue
                        # 字符串转为整型苏剧
                        if "0x" in item_str[: offset]:
                            min_int = int(item_str[: offset], 16)
                        else:
                            min_int = int(item_str[: offset])
                        if "0x" in item_str[offset + 1:]:
                            max_int = int(item_str[offset + 1:], 16)
                        else:
                            max_int = int(item_str[offset + 1:])
                        if "0x" in value:
                            value = int(value, 16)
                        else:
                            value = int(value)
                        if value < min_int or value > max_int:
                            logger.error(self.tableWidget.item(row, 0).text(), value, item_str)
                            continue

            # 从tableWidget item读取数据，更新ddr params
            for row in range(self.tableWidget.rowCount()):
                param_name = self.tableWidget.item(row, 0).text()
                try:
                    param_value = self.tableWidget.item(row, 1).text()
                except AttributeError:
                    continue
                self.res_dict[param_name] = param_value
    # 定义dram type相关按钮
    def clicked_button_dram_type(self, dram_type):
        if self.res_dict:
            self.update_ddr_params_by_tableWidget()
            self.show_ddr_params_tableWidget(self.ddrbin_param_dict[self.chip_name], dram_type)
            if self.language == "English":
                self.statusBar().showMessage(str('OK: get ' + dram_type + ' parameter succeed，Creat ddr bin if completed modification.'))
            else:
                self.statusBar().showMessage(str('OK: 获取' + dram_type + '参数成功，请完成配置完成后点击对应的【生成】按钮。'))
        else:
            if self.language == "English":
                self.statusBar().showMessage('Error: DDR parameter is NULL，please reselect ddr bin which you need modify.')
            else:
                self.statusBar().showMessage('Error: DDR配置参数为空，请先选择要修改原始的文件。')
    def clicked_button_common(self):
        self.clicked_button_dram_type("COMMON")
    def clicked_button_lpddr5(self):
        self.clicked_button_dram_type("LPDDR5")
    def clicked_button_lpddr4(self):
        self.clicked_button_dram_type("LPDDR4")
    def clicked_button_lpddr4x(self):
        self.clicked_button_dram_type("LPDDR4X")
    def clicked_button_lpddr3(self):
        self.clicked_button_dram_type("LPDDR3")
    def clicked_button_lpddr2(self):
        self.clicked_button_dram_type("LPDDR2")
    def clicked_button_ddr4(self):
        self.clicked_button_dram_type("DDR4")
    def clicked_button_ddr3(self):
        self.clicked_button_dram_type("DDR3")
    def clicked_button_ddr2(self):
        self.clicked_button_dram_type("DDR2")
    def clicked_button_all(self):
        self.clicked_button_dram_type("DRAM_TYPE_ALL")
    # 定义按钮功能，选取ddr bin文件
    def clicked_button_db_sel(self):
        openfile_name = QtWidgets.QFileDialog.getOpenFileName(self, '选择文件', '', 'binary file(*.bin);;All Files(*)')
        self.lineEdit_ddrbin.setText(str(openfile_name[0]))
        self.clicked_button_db_con()
    # 定义按钮功能，读取ddr bin中的ddr params
    def clicked_button_db_con(self):
        # 芯片型号和语言选项框配置为不可用，Reset UI配置后才可重新选择芯片型号和语言
        self.comboBox_chipName.setEnabled(False)
        self.comboBox_language.setEnabled(False)

        if self.chip_name == "Other":
            self.chip_other = self.lineEdit_chipOther.text()
            if len(self.chip_other) == 0:
                if self.language == "English":
                    self.show_warning_message("Select CHIP is Other, please input chip info.")
                else:
                    self.show_warning_message("选择的CHIP为Other，请手动输入芯片型号。")
                return
        else:
            self.chip_other = ""

        self.res_dict = {}
        # 读取输入框内的ddr bin文件路径
        self.ddr_bin_file_path = self.lineEdit_ddrbin.text()

        # 如果输入的是“loader”，则根据平台支持情况，显示loader相关按钮或者提示不支持。
        if self.ddr_bin_file_path == 'loader':
            if self.chip_name in self.loader_pack_dict:
                self.loader_pack_params = self.loader_pack_dict[self.chip_name]
                # 显示loader相关功能按钮
                self.pushButton_ld_gen.setVisible(True)
                #self.pushButton_ld_con.setVisible(True)
                #self.pushButton_ld_sel.setVisible(True)
                #self.lineEdit_loader.setVisible(True)
                self.lineEdit_pack_param.setVisible(True)
                # button disable，ddr bin相关按钮
                self.pushButton_db_sel.setEnabled(False)
                self.pushButton_db_con.setEnabled(False)
                # button disable，loader生成按钮，在加载loader文件后enable
                self.pushButton_ld_gen.setEnabled(False)
            else:
                if self.language == "English":
                    self.show_warning_message("[" + self.chip_name + "]，unsupported.")
                else:
                    self.show_warning_message("【" + self.chip_name + "】，暂不支持该功能。")
            return
        elif self.ddr_bin_file_path == 'all_param':
            self.pushButton_all.setVisible(True)
            return

        # 检查ddr bin文件是否存在
        if os.path.exists(self.ddr_bin_file_path) == False:
            if self.language == "English":
                self.statusBar().showMessage('Error: file does not exist，please check file path.')
            else:
                self.statusBar().showMessage('Error：文件不存在，请检查文件路径是否正确。')
            return
        # 如果是loader文件显示loader相关按钮，否则隐藏loader相关按钮
        if modify_ddr_bin.check_if_loadr_file(self.ddr_bin_file_path) == True:
            if self.chip_name in self.loader_pack_dict:
                self.loader_pack_params = self.loader_pack_dict[self.chip_name]
            else:
                if self.language == "English":
                    self.show_warning_message("[" + self.chip_name + "]，loader file unsupported.")
                else:
                    self.show_warning_message("【" + self.chip_name + "】，不支持加载loader文件。")
                return
            #self.show_warning_message("所选文件可能是loader文件，请确认，并正确选择ddr bin文件。")
            # 显示loader相关功能按钮
            self.pushButton_ld_gen.setVisible(True)
            self.lineEdit_pack_param.setVisible(True)

            #self.lineEdit_loader.setText(self.ddr_bin_file_path)
            self.clicked_button_ld_con()
            return
        else:
            # 显示loader相关功能按钮
            self.pushButton_ld_gen.setVisible(False)
            self.lineEdit_pack_param.setVisible(False)

        # 读取ddr bin文件中的ddr prams
        self.res_dict, ret_int = modify_ddr_bin.get_ddr_params(self.ddr_bin_file_path, self.chip_name, self.chip_other)
        if self.radioButton_saveCfg.isChecked():
            for key in self.res_dict_diff:
                self.res_dict[key] = self.res_dict_diff[key]

        if ret_int == -1:
            if self.language == "English":
                self.show_warning_message("please check file is [" + self.chip_name + "]" + self.chip_other + " loader, reselect if not.")
            else:
                self.show_warning_message("请确认所选文件是否为【" + self.chip_name + "】" + self.chip_other + " loader文件。如果不是，请重选。")
        if ret_int == -2:
            self.show_error_message("ddrbin_tool chip name error")
            return
        if ret_int == -3:
            self.show_error_message("ddrbin_tool does not exist")
            return

        if not bool(self.res_dict):
            if self.language == "English":
                self.statusBar().showMessage('ERROR: get DDR parameter fail, please check file is ddr bin.')
                self.show_error_message("get DDR parameter fail, please check file is ddr bin.")
            else:
                self.statusBar().showMessage('ERROR： 获取DDR参数，失败。请检查所选文件是否为ddr bin文件。')
                self.show_error_message("获取DDR参数，失败。请检查所选文件是否为ddr bin文件。")
        else:
            if self.language == "English":
                self.statusBar().showMessage('OK：get DDR prameter succeed, please select dram type which need modify.')
            else:
                self.statusBar().showMessage('OK：获取DDR参数，成功。请选择要修改参数的DDR类型。')
        # tableWidget默认显示“COMMON”属性参数
        self.show_ddr_params_tableWidget(self.ddrbin_param_dict[self.chip_name], "COMMON")
        # 根据ddr params中的dram type支持情况，使能相关按钮
        self.enable_pushbutton_dram_type(self.res_dict)
    # 定义按钮功能，生成ddr bin
    def clicked_button_db_gen(self):
        if self.res_dict:
            self.update_ddr_params_by_tableWidget(remove_CellWidget = False)
            file_path = modify_ddr_bin.update_ddr_bin(self.res_dict, self.ddr_bin_file_path, self.chip_name, self.chip_other, not self.radioButton_overWrite.isChecked())
            if self.language == "English":
                self.statusBar().showMessage(str('OK：Create ddr bin succeed. file path: ' + file_path))
            else:
                self.statusBar().showMessage(str('OK：生成ddr bin，成功。文件路径为：' + file_path))
            self.res_dict_diff, data_str =  modify_ddr_bin.cpmpare_ddr_param()
            if self.language == "English":
                self.show_information_message('Crete ddr bin succeed, file path: ' + file_path + '\n\n' + data_str)
            else:
                self.show_information_message('生成ddr bin，成功。文件路径为：' + file_path + '\n\n' + data_str)
        else:
            if self.language == "English":
                self.statusBar().showMessage('Error: DDR parameter is NULL.')
            else:
                self.statusBar().showMessage('Error: DDR配置参数为空，请先读取')
    # 定义按钮功能，选取loader文件
    #def clicked_button_ld_sel(self):
    #    openfile_name = QtWidgets.QFileDialog.getOpenFileName(self, '选择文件', '', 'binary file(*.bin);;All Files(*)')
        #self.lineEdit_loader.setText(str(openfile_name[0]))
    #    self.clicked_button_ld_con()
    # 定义按钮功能，读取loader的ddr params
    def clicked_button_ld_con(self):
        self.res_dict = {}
        # 读取输入框内的loader文件路径
        #self.loader_file_path = self.lineEdit_loader.text()
        self.loader_file_path = self.ddr_bin_file_path

        if self.loader_file_path == 'pack_param':
            self.lineEdit_pack_param.setVisible(True)
            return

        # 检查loader文件是否存在
        if os.path.exists(self.loader_file_path) == False:
            if self.language == "English":
                self.statusBar().showMessage('Error: loader file does not exist')
            else:
                self.statusBar().showMessage('Error：loader文件不存在')
            return

        # loader文件解包
        self.loader_params_dict = modify_ddr_bin.loader_unpack(self.loader_file_path, self.chip_name)
        if not bool(self.loader_params_dict):
            if self.language == "English":
                self.statusBar().showMessage('Error: loader unpack fail, please check file is loader.')
                self.show_error_message("loader unpack fail, please check file is loader.")
            else:
                self.statusBar().showMessage('Error：解包失败，请检查该文件是否为loader文件。')
                self.show_error_message("解包失败，请检查该文件是否为loader文件。")
            return
        # 检查boot_merge文件是否存在
        if self.loader_params_dict['boot_merge_tool_path'] == "null":
            self.show_error_message("boot_merge does not exist.")
            return

        # 记录loader文件路径
        self.loader_file_path = self.loader_params_dict["loader_file_path"]
        # 记录ddr bin文件路径
        self.ddr_bin_file_path = self.loader_params_dict["loader_471"]
        # 读取ddr bin文件中的ddr params
        self.res_dict, ret_int = modify_ddr_bin.get_ddr_params(self.ddr_bin_file_path, self.chip_name, self.chip_other)
        if self.radioButton_saveCfg.isChecked():
            for key in self.res_dict_diff:
                self.res_dict[key] = self.res_dict_diff[key]

        if ret_int == -1:
            if self.language == "English":
                self.show_warning_message("please check file is [" + self.chip_name + "]" + self.chip_other + " loader, reselect if not.")
            else:
                self.show_warning_message("请确认所选文件是否为【" + self.chip_name + "】" + self.chip_other + " loader文件。如果不是，请重选。")
        if ret_int == -2:
            self.show_error_message("ddrbin_tool chip name error")
            return
        if ret_int == -3:
            self.show_error_message("ddrbin_tool does not exist")
            return

        # tableWidget默认显示“COMMON”属性参数
        self.show_ddr_params_tableWidget(self.ddrbin_param_dict[self.chip_name], "COMMON")
        # 根据ddr params中的dram type支持情况，使能相关按钮
        self.enable_pushbutton_dram_type(self.res_dict)
        if self.language == "English":
            self.statusBar().showMessage('OK: get DDR parameter succeed, please select dram type which need modify')
        else:
            self.statusBar().showMessage('OK： 获取DDR参数，成功。请选择要配置的DDR类型。')

        self.loader_params_dict["pack_params"] += ' ' + self.loader_pack_params
        self.lineEdit_pack_param.setText(self.loader_params_dict["pack_params"])
    #定义按钮功能，重新生成loader
    def clicked_button_ld_gen(self):
        # loader警告框
        if self.language == "English":
            self.show_warning_message("The loader file generated by the tool may have different packing parameters from the original loader file. " + \
                "Therefore, the Loader generated by the tool is only use for debugging DDR parameter. " +\
                "After DDR parameter debugging, select create the DDR bin file and pack the generated DDR bin file in the code project.")
        else:
            self.show_warning_message("该工具重新生成的loader文件与原始loader文件可能存在打包参数的差异，因此通过该工具生成的loader仅供调测DDR参数使用。" +\
                "正式使用的loader，请选择生成ddr bin文件，并将生成的ddr bin文件放入代码工程中进行打包。")
        if self.res_dict:
            self.update_ddr_params_by_tableWidget(remove_CellWidget = False)
            self.loader_params_dict["loader_471"] = \
                modify_ddr_bin.update_ddr_bin(self.res_dict, self.ddr_bin_file_path, self.chip_name, self.chip_other, False)
            self.loader_params_dict["pack_params"] = self.lineEdit_pack_param.text()

            if "loader_file_path" not in self.loader_params_dict:
                if self.language == "English":
                    self.statusBar().showMessage('Error: Create loader fail')
                else:
                    self.statusBar().showMessage('Error： 生成loader，失败。请先加载loader文件。')
                return
            # 打包loader文件
            file_path = modify_ddr_bin.loader_pack(self.loader_params_dict, not self.radioButton_overWrite.isChecked())
            if self.language == "English":
                # 检查loader文件是否成功生成
                if os.path.exists(file_path):
                    self.statusBar().showMessage('OK: Create loader succeed, file path：' + file_path)
                else:
                    self.statusBar().showMessage('ERROR: Create loader, failed.')
            else:
                # 检查loader文件是否成功生成
                if os.path.exists(file_path):
                    self.statusBar().showMessage('OK: 生成loader，成功。文件路径为：' + file_path)
                else:
                    self.statusBar().showMessage('ERROR: 生成loader，失败。')
            self.res_dict_diff, data_str =  modify_ddr_bin.cpmpare_ddr_param()
            if self.language == "English":
                # 检查loader文件是否成功生成
                if os.path.exists(file_path):
                    self.show_information_message('Create loader succeed, file path：' + file_path + '\n\n' + data_str)
                else:
                    self.show_error_message('ERROR: Create loader, failed.')
            else:
                # 检查loader文件是否成功生成
                if os.path.exists(file_path):
                    self.show_information_message('生成loader，成功。文件路径为：' + file_path + '\n\n' + data_str)
                else:
                    self.show_error_message('ERROR: 生成loader，失败。')
        else:
            if self.language == "English":
                self.statusBar().showMessage('Error: DDR parameter is NULL')
            else:
                self.statusBar().showMessage('Error: DDR配置参数为空，请先读取')

if __name__=="__main__":
    #QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_EnableHighDpiScaling)
    app = QtWidgets.QApplication(sys.argv)
    # 显示登录界面
    first = login_ddrbin()
    first.show()

    sys.exit(app.exec())
